/*
* multi_curve.h
* DIN Is Noise is copyright (c) 2006-2025 Jagannathan Sampath
* DIN Is Noise is released under GNU Public License 2.0
* For more information, please visit https://dinisnoise.org/
*/


#ifndef __MULTICURVE__
#define __MULTICURVE__

#include <string>
#include <fstream>
#include <vector>

#include "point.h"
#include "curve.h"
#include "box.h"

typedef std::vector< pointinfo<float> > points_array;

struct multi_curve {
	
	// multiple bezier curves
	//

	std::vector<curve> curv; // bezier curves of this multi_curve
  int ncurvs;

  points_array vertices;
  int num_vertices; // same number of left and right tangents
  int last_vertex;

  points_array left_tangents;
  points_array right_tangents;

  multi_curve ();
  multi_curve (const std::string& filename); // load from file

	// vertex ops
  int set_vertex (int i, float x, float y);
  void add_vertex (float x, float y, int p = 0);
	int insert (float x, float y);
	int remove (int i); 
  void get_vertex (int i, float& x, float& y);

	// tangent ops
  void add_left_tangent (float x, float y, int p = 0);
  void add_right_tangent (float x, float y, int p = 0);
  int set_left_tangent (int i, float x, float y);
  int set_right_tangent (int i, float x, float y);
	void get_left_tangent (int i, float& x, float& y);
	void get_right_tangent (int i, float& x, float& y);
  double get_tangent_mag (int i, points_array& pa);
  double get_left_tangent_mag (int i);
  double get_right_tangent_mag (int i);
  void unfoldx_left_tangents ();
  void unfoldy_left_tangents ();
  void unfoldx_right_tangents ();
  void unfoldy_right_tangents ();
  void unfold_left_tangent (int id, int w);
  void unfold_right_tangent (int id, int w);

  // adds
  void add (multi_curve& c);
  void add_vertex_tangents (point<float>& v, point<float>& lt, point<float>& rt);

	float limit; // same for all curves (see curve.h)
	void set_limit (float d);

  int shapeform; // shapeform?
  void set_shapeform (int yesno);
	void check_shapeform ();
	void get_xy (double d, float& x, float& y); 

  // profile of multi_curve ie of all bezier curves
  void require_eval (); 
  void evaluate (); 
	int get_profile_points (std::vector<crvpt>& profile); // all curves
	std::vector<crvpt>& get_profile_points (int i); // for ith curve
  int get_total_points ();
	int get_total_points (std::vector<int>& crvs);

	// xform
  point<float> cen;
  void setcentroid ();
	void calc_bbox (box<float>& b);
  void rotate (float angle);
  void scale (float sx, float sy);
  void move (float dx, float dy);

	// visual
	float r, g, b; // color of profile
	float rt, gt, bt; // color of tangents
  void set_color (); // random color
  void set_color (float rr, float gg, float bb);
  void get_color (float& rr, float& gg, float& bb) { rr = r; gg = g; bb = b;}
  void calc_tangent_color ();

	// utils
  std::string name; // name
	void set_name (const std::string& name);
	void load (const std::string& filename);
	void load (std::ifstream& file);
	void save (const std::string& filename);
	void save (std::ofstream& file);
  void clear (int all = 1); // all => color & name

};

void create_polyline (multi_curve& crv, const points_array& pts);
void convert2_polyline (multi_curve& crv, int what);
void convert2_catmull_rom (multi_curve& crv, float tangent_size);
multi_curve* get_curve (const std::string& name);
multi_curve* check_list (multi_curve** lst, int n, const std::string& name);
void paste_append (multi_curve& to, multi_curve& from, int alt = 0);

#endif
