/*
* delay.cc
* DIN Is Noise is copyright (c) 2006-2026 Jagannathan Sampath
* DIN Is Noise is released under GNU Public License 2.0
* For more information, please visit https://dinisnoise.org/
*/


#include "delay.h"
#include "utils.h"
#include "log.h"
#include <string.h>
#include <stdlib.h>
#include <string>
using namespace std;

extern int SAMPLE_RATE;

void delay::zero () {
  maxsamples = 0;
  nsamples = 0;
  samples = 0;
  fbk = 0;
  vol = 0;
  id = 0;
  msecs = 0;
  beats = 1.0f;
  bps = 1.0f;
}

delay::delay (const string& fn, const string& vn) {
  fbk_fname = fn;
  vol_fname = vn;
  fbk_crv.load (fbk_fname);
  vol_crv.load (vol_fname);
  fbk_sol (&fbk_crv);
  vol_sol (&vol_crv);
  fbk_lis (this, &fbk_sol, fbk);
  vol_lis (this, &vol_sol, vol);
  zero ();
}

delay::~delay() {

  dlog << "--- deleting delay buffers ---" << endl;
  float* bf [3] = {samples, fbk, vol};
  const char* nm [3] = {"samples", "feedback", "volume"};
  for (int i = 0; i < 3; ++i) {
    if (bf[i]) {
      dlog << "  deleting: " << nm[i] << endl;
      free (bf[i]);
    }
  }
  dlog << "+++ deleted delay buffers +++" << endl;

  fbk_crv.save (fbk_fname);
  vol_crv.save (vol_fname);


}

void delay::prep () {

  static const float thou_1 = 1.0 / 1000.0f;
  static const int size_of_float = sizeof (float);

  int n = (int) (msecs * SAMPLE_RATE * thou_1 + 0.5);
  if (n < 1) n = 1;

  if (n > maxsamples) {
    float* newsamples = (float *) calloc (n, size_of_float);
    if (nsamples) {
      for (int j = 0; j < nsamples; ++j) newsamples[j] = samples[j];
      free (samples);
    }
    samples = newsamples;
    int extra = n - nsamples;
    fill (samples, samples [nsamples - 1], samples [0], extra);
    float** nbf[2] = {&fbk, &vol};
    for (int i = 0; i < 2; ++i) {
      float* bi = *nbf[i];
      if (bi) free (bi);
      *nbf[i] = (float *) calloc (n, size_of_float);
    }
    maxsamples = n;
  }   
  nsamples = n;
  clamp<int>(0, id, nsamples - 1);
}

void delay::set (float _beats) {
  beats = _beats;
  set (beats, bps);
}

void delay::set (float _beats, float _bps) {
  if (_beats > 0) beats = _beats; else beats = 1.0f;
  bps = _bps;
  msecs = 1000.0f * beats / bps;
  prep ();
  float dx = 1.0f / (nsamples - 1), x = -dx;
  fbk_sol (x, dx, nsamples, fbk);
  vol_sol (x, dx, nsamples, vol);
  fbk_lis.result = fbk;
  vol_lis.result = vol;
}

void delay::get (float& p, float& q) {
  p = beats;
  q = bps;
}

void delay::operator() (float* out, int n, float fdr) {
  float fo, fi;
  for (int i = 0; i < n; ++i) {
    float& si = samples [id];
    fo = fbk [id] * si;
    fi = out [i] + fo;
    si = fi;
    out[i] += (fdr * vol[id] * fo);
    if (++id < nsamples) ; else id = 0;
  }
}
